//--------------------------------------------------------------------------------
// skeleton.cc
//
// This is a skeleton program to read/analyse BPYS5 Ntuples via standalnoe c++
// If you have installed root, you can compile this file as follows:
// g++ skeleton.cc -o skeleton.exe `root-config --cflags --glibs`
//
// version 0.1
// Generated Ahmet.Bingul@cern.ch (27 Sep 2024)
// Modified  Ahmet.Bingul@cern.ch (08 Oct 2024)
//--------------------------------------------------------------------------------

// C++ headers
#include <iostream>
#include <fstream>
#include <cmath>
#include <vector>
#include <iomanip>
using namespace std;

// ROOT headers
#include "TF1.h"
#include "TH1F.h"
#include "TH2F.h"
#include "TH1D.h"
#include "TFile.h"
#include "TTree.h"
#include "TStyle.h"
#include "TChain.h"
#include "TNtuple.h"
#include "TCanvas.h"
#include "TVector3.h"
#include "TApplication.h"
#include "Math/Vector4D.h"
#include "TLorentzVector.h"

// MakeFile headers
#include "include/BplusBestChi.C"
#define GeV 1e-3

int main()
{
  bool   isMC          = false;   // MC or data
  double eventFraction = 1.0;     // e.g. 0.2 means take 20% of the total events
  double CptBplus      = 2000.0;  // momentum cut for B+
  double CptKminus     = 2500.0;  // momentum cut for K-
  double CChi2Ndf      = 3.0;     // chi2 / nof
  double CdR           = 1.0;     // dR(B+,K-)

  // pdg masses
  double const MBpmPDG  = 5279.41;
  double const MkaonPDG = 493.677;
  double const MmuonPDG = 105.6583755;
  //double const MJpsiPDG = 3096.9;

  cout << "INFO\t" << "*****************************"<< endl;
  cout << "INFO\t" << "*** Bs* -> B+ K- Analysis ***" << endl;
  cout << "INFO\t" << "*****************************"<< endl;
  cout << "INFO\t" << endl;
  if(isMC) cout << "INFO\t" << "MC NTuples will be used." << endl;
  else     cout << "INFO\t" << "Real data Ntuples will be used." << endl;
  cout << "INFO\t" << endl;
  cout << "INFO\t" << "CptBplus   = " << CptBplus  << endl;
  cout << "INFO\t" << "CptKminus  = " << CptKminus  << endl;
  cout << "INFO\t" << "CdR        = " << CdR  << endl;

  // input and output root file settings
  string inputFile, outputFile, signalFile = "none";
  if(isMC){
    inputFile  = "/eos/atlas/atlascerngroupdisk/phys-beauty/BsJpsiPhi/BsExciteMC/mc16_13TeV.800827.Bd_Bplus.e8307_a875_r10724.v4.root"; // en son bunu kullanmıştık
    outputFile = "HistoMC.root";
  }
  else{
    inputFile  = "/eos/atlas/atlascerngroupdisk/phys-beauty/BsJpsiPhi/run2BplusBsExcite/data15_16_17_18_BsExcite.Bplus.root"; // all Run2
    outputFile = "HistoDA.root";
  }
  cout << "INFO\t" << "inputFile  = " << inputFile  << endl;
  cout << "INFO\t" << "outputFile = " << outputFile << endl;
  if(isMC) cout << "INFO\t" << "signalFile = " << signalFile << endl;

  cout << "INFO\t" << "Setting up ROOT objects ..." << endl;
  gROOT->SetBatch();

  // chains to read tree in data files
  TChain *chData = new TChain("BplusBestChi","BplusBestChi"); // key name
  chData->Add(inputFile.c_str());

  // Define key pointer
  BplusBestChi *sample = new BplusBestChi(chData);

  // total number events to process
  Long64_t nEvents = Long64_t(chData->GetEntries() * eventFraction);
  cout << "INFO\t" << "Total events desired = " << nEvents << " ("<< round(eventFraction*100.0)<<"% of total events)" << endl;
  cout << "INFO\t" << endl;

  // root file
  TFile *rootFile = new TFile(outputFile.c_str(),"recreate");

  // histograms
  TH1F *h_m_Bplus    = new TH1F("h_m_Bplus",   ";Mass M(Jpsi,K+) [MeV];Entries", 400, 5000, 6000);
  TH1F *h_pt_Bplus   = new TH1F("h_pt_Bplus",  ";pt [MeV];Entries",              700, 0,    70000);
  TH1F *h_eta_Bplus  = new TH1F("h_eta_Bplus", ";eta ;Entries",                  300, -4,   4);
  TH1F *h_phi_Bplus  = new TH1F("h_phi_Bplus", ";phi ;Entries",                  200, -3.2, 3.2);

  TH1F *h_m_BK       = new TH1F("h_m_BK",   ";M(B+,K-) [MeV];Entries", 200,5830,5950);

  TH1F *h_pt_BK_SigReg  = new TH1F("h_pt_BK_SigReg",  ";pt;Entries",  700, 0, 70000);
  TH1F *h_eta_BK_SigReg = new TH1F("h_eta_BK_SigReg", ";eta;Entries", 30, -3, 3);
  TH1F *h_phi_BK_SigReg = new TH1F("h_phi_BK_SigReg", ";phi;Entries", 30, -3.2, 3.2);

  TH1F *h_pt_BK_SideReg = new TH1F("h_pt_BK_SideReg", ";pt;Entries",  700, 0, 70000);
  TH1F *h_eta_BK_SideReg= new TH1F("h_eta_BK_SideReg",";eta;Entries", 30, -3, 3);
  TH1F *h_phi_BK_SideReg= new TH1F("h_phi_BK_SideReg",";phi;Entries", 30, -3.2, 3.2);

  TH1F *h_pt_Kminus_SigReg  = new TH1F("h_pt_Kminus_SigReg", ";pt (K-) [MeV];Entries", 500, 0,  15000);
  TH1F *h_eta_Kminus_SigReg = new TH1F("h_eta_Kminus_SigReg",";eta (K-);Entries",      30, -3,  3);
  TH1F *h_phi_Kminus_SigReg = new TH1F("h_phi_Kminus_SigReg",";phi (K-);Entries",      30, -3.2,3.2);

  TH1F *h_pt_Kminus_SideReg  = new TH1F("h_pt_Kminus_SideReg", ";pt (K-) [MeV];Entries", 500, 0,  15000);
  TH1F *h_eta_Kminus_SideReg = new TH1F("h_eta_Kminus_SideReg",";eta (K-);Entries",      30, -3,  3);
  TH1F *h_phi_Kminus_SideReg = new TH1F("h_phi_Kminus_SideReg",";phi (K-);Entries",      30, -3.2,3.2);

  TH1F *h_pt_muons_SigReg  = new TH1F("h_pt_muons_SigReg", ";pt (K-) [MeV];Entries", 500, 0,  15000);
  TH1F *h_eta_muons_SigReg = new TH1F("h_eta_muons_SigReg",";eta (K-);Entries",      30, -3,  3);
  TH1F *h_phi_muons_SigReg = new TH1F("h_phi_muons_SigReg",";phi (K-);Entries",      30, -3.2,3.2);

  TH1F *h_pt_muons_SideReg  = new TH1F("h_pt_muons_SideReg", ";pt (K-) [MeV];Entries", 500, 0,  15000);
  TH1F *h_eta_muons_SideReg = new TH1F("h_eta_muons_SideReg",";eta (K-);Entries",      30, -3,  3);
  TH1F *h_phi_muons_SideReg = new TH1F("h_phi_muons_SideReg",";phi (K-);Entries",      30, -3.2,3.2);

  TH1F *h_dR_BplusKminus_SigReg  = new TH1F("h_dR_BplusKminus_SigReg", ";dR (B+,K-);Entries", 100, 0, 1);
  TH1F *h_dR_BplusKminus_SideReg  = new TH1F("h_dR_BplusKminus_SideReg", ";dR (B+,K-);Entries", 100, 0, 1);

  TH1F *h_cosThetaStar_SigReg  = new TH1F("h_cosThetaStar_SigReg", ";dR (B+,K-);Entries", 100, -1, 1);
  TH1F *h_cosThetaStar_SideReg  = new TH1F("h_cosThetaStar_SideReg", ";dR (B+,K-);Entries", 100, -1, 1);

  TH1F *h_PB_over_PK_SigReg  = new TH1F("h_PB_over_PK_SigReg", ";P(B+)/P(K-);Entries", 100, 0, 40);
  TH1F *h_PB_over_PK_SideReg  = new TH1F("h_PB_over_PK_SideReg",";P(B+)/P(K-);Entries", 100, 0, 40);

  // start of main event loop ------------------------------------------------------
  for(Long64_t ev = 0; ev < nEvents; ev++)
  {
    sample->GetEntry(ev);     //get entries in the event (either MC or DA)
    if ( (ev % (nEvents/100)) == 0) cout << "TIME\t" << "Processed " << ev << " events [" << 100*ev/nEvents << "%]" << endl;

    // //////////////////////////////////////
    // B+ -> Jpsi + K+ Selection
    // //////////////////////////////////////

    if(sample->B_chi2_ndof > CChi2Ndf) continue;

    // fill Bmass without applying any cut
    double bmass   = sample->B_mass;
    int    bcharge = sample->B_trk_charge;

    bool goodBplus = (bmass > 5180. && bmass < 5420.) &&
                      bcharge != 0 &&
                      sample->B_pT > CptBplus &&
                      sample->B_tau_MinA0ConstM > 0.2;
    if( !goodBplus ) continue;

    // jpsi, K+ and and B+ candidates
    ROOT::Math::PtEtaPhiMVector mu1  {sample->B_mu1_pT, sample->B_mu1_eta, sample->B_mu1_phi, MmuonPDG};
    ROOT::Math::PtEtaPhiMVector mu2  {sample->B_mu2_pT, sample->B_mu2_eta, sample->B_mu2_phi, MmuonPDG};
    ROOT::Math::PtEtaPhiMVector kaon1{sample->B_trk_pT, sample->B_trk_eta, sample->B_trk_phi, MkaonPDG};
    auto jpsi  = mu1 + mu2;
    auto Bplus = jpsi + kaon1;

    // B+ mass after cuts
    h_m_Bplus->Fill(bmass);
    h_pt_Bplus->Fill(Bplus.Pt());
    h_eta_Bplus->Fill(Bplus.Eta());
    h_phi_Bplus->Fill(Bplus.Phi());

    // //////////////////////////////////////
    // Bs* -> B+ + K- Selection
    // //////////////////////////////////////

    // check if there is a Bs* candidate
    unsigned int nBsCand = sample->Bs_ex_mass->size();
    if(nBsCand == 0) continue;

    // Find the best K- (closest to B+) from PV
    int kaonIndex = -1;
    double dRbest = 1.0e99;
    if(nBsCand==1 && sample->Bs_ex_trk_DeltaR->at(0) < CdR){
        kaonIndex = 0;
        dRbest = sample->Bs_ex_trk_DeltaR->at(0);
    }
    else if(nBsCand>1){
      for(unsigned int i = 0; i < nBsCand; i++){
         double dR = sample->Bs_ex_trk_DeltaR->at(i);
         if(dR > CdR ) continue;
         if(dR < dRbest){
           dRbest = dR;
           kaonIndex = i;
         }
      }
    }

    // apply cuts for K- selection
    if(kaonIndex == -1) continue; // K- is not close enough to Bs*
    if(fabs(sample->Bs_ex_trk_extrapd0_minA0->at(kaonIndex)) > 0.05) continue; // transverse impact parameter (d0) cut
    if(sample->Bs_ex_trk_pt->at(kaonIndex) < CptKminus ) continue; // transverse momentum cut
    if(sample->B_trk_charge * sample->Bs_ex_trk_charge->at(kaonIndex) > 0 ) continue; // we must have opposite charges

    // Now build K- candidate
    ROOT::Math::PtEtaPhiMVector kaon2{sample->Bs_ex_trk_pt->at(kaonIndex),
                                      sample->Bs_ex_trk_eta->at(kaonIndex),
                                      sample->Bs_ex_trk_phi->at(kaonIndex),
                                      MkaonPDG};

    // Bs* candidates
    double m     = sample->Bs_ex_mass->at(kaonIndex);  // unconstrained
    auto   Bstar = Bplus + kaon2;                      // unconstrained

    // Go to center of mass frame of Bs*.
    // Evaluate angle between momentum vectors of K- "in CM frame" and Bs* "in lab frame".
    TLorentzVector Bss, Kss;
    Kss.SetPxPyPzE(kaon2.Px(),kaon2.Py(),kaon2.Pz(),kaon2.E()); // K-  in lab
    Bss.SetPxPyPzE(Bstar.Px(),Bstar.Py(),Bstar.Pz(),Bstar.E()); // Bs* in lab (actually copy of Bstar)
    Kss.Boost(-Bstar.Px()/Bstar.E(), -Bstar.Py()/Bstar.E(), -Bstar.Pz()/Bstar.E()); // K- in CM
    double cosTheta_BstarKminus = cos(Bss.Angle(Kss.Vect()));
    double PB_over_PK = Bplus.P()/kaon1.P();
    //if(cosTheta_BstarKminus < -0.8) continue;

    // same/opposite charge
    if(sample->Bs_ex_trk_chargeOpposite->at(kaonIndex))
    {
       double m0 = m + MBpmPDG - bmass; // mass difference
       h_m_BK->Fill(m0);

       if(m0>5830.0 && m0<5850.0){        // Signal region
          h_pt_BK_SigReg ->Fill(Bstar.Pt());
          h_eta_BK_SigReg->Fill(Bstar.Eta());
          h_phi_BK_SigReg->Fill(Bstar.Phi());

          h_pt_Kminus_SigReg ->Fill(kaon2.Pt());
          h_eta_Kminus_SigReg->Fill(kaon2.Eta());
          h_phi_Kminus_SigReg->Fill(kaon2.Phi());

          h_pt_muons_SigReg ->Fill(mu1.Pt());
          h_eta_muons_SigReg->Fill(mu1.Eta());
          h_phi_muons_SigReg->Fill(mu1.Phi());
          h_pt_muons_SigReg ->Fill(mu2.Pt());
          h_eta_muons_SigReg->Fill(mu2.Eta());
          h_phi_muons_SigReg->Fill(mu2.Phi());

          h_dR_BplusKminus_SigReg->Fill(dRbest);
          h_cosThetaStar_SigReg->Fill(cosTheta_BstarKminus);
          h_PB_over_PK_SigReg->Fill(PB_over_PK);
       }
       if((m0>5860.0 && m0<5880.0) || (m0>5810.0 && m0<5821.0) ){ // sideband region
          h_pt_BK_SideReg ->Fill(Bstar.Pt());
          h_eta_BK_SideReg->Fill(Bstar.Eta());
          h_phi_BK_SideReg->Fill(Bstar.Phi());

          h_pt_Kminus_SideReg ->Fill(kaon2.Pt());
          h_eta_Kminus_SideReg->Fill(kaon2.Eta());
          h_phi_Kminus_SideReg->Fill(kaon2.Phi());

          h_pt_muons_SideReg ->Fill(mu1.Pt());
          h_eta_muons_SideReg->Fill(mu1.Eta());
          h_phi_muons_SideReg->Fill(mu1.Phi());
          h_pt_muons_SideReg ->Fill(mu2.Pt());
          h_eta_muons_SideReg->Fill(mu2.Eta());
          h_phi_muons_SideReg->Fill(mu2.Phi());

          h_dR_BplusKminus_SideReg->Fill(dRbest);
          h_cosThetaStar_SideReg->Fill(cosTheta_BstarKminus);
          h_PB_over_PK_SideReg->Fill(PB_over_PK);
       }
    } // end if opposite charge

  }
  //end of event main loop
  // ------------------------------------------------------------------------
  cout << "INFO\t" << endl;
  cout << "INFO\t" << "Main event loop has been finished." << endl;
  cout << "INFO\t" << "Now, finalizing the program ..." << endl;

  // write all histograms to an output histogram file defined above
  rootFile->Write();
  rootFile->Close();

  //TCanvas *c1 = new TCanvas("c1","c1",800,600);
  //hm_Bstar->Draw();
  //c1->Print("hm_Bstar.png");

  cout << "INFO\t" << "ALL OK." << endl;
  cout << "INFO\t" << endl;

  return 0;
}
// EOF
